/*
 * Copyright 2002-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.integration.samples.helloworld;

import org.springframework.context.ApplicationContext;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.integration.channel.BeanFactoryChannelResolver;
import org.springframework.integration.channel.ChannelResolver;
import org.springframework.integration.channel.PollableChannel;
import org.springframework.integration.core.MessageChannel;
import org.springframework.integration.message.StringMessage;

/**
 * Demonstrates a basic Message Endpoint that simply prepends a greeting
 * ("Hello ") to an inbound String payload from a Message. This is a very
 * low-level example, using Message Channels directly for both input and
 * output. Notice that the output channel has a queue sub-element. It is
 * therefore a PollableChannel and its consumers must invoke receive() as
 * demonstrated below. The {@link BeanFactoryChannelResolver} is used here
 * rather than performing a generic dependency lookup from the context.
 * <p>
 * View the configuration of the channels and the endpoint (a &lt;service-activator/&gt;
 * element) in 'helloWorldDemo.xml' within this same package.
 * 
 * @author Mark Fisher
 * @author Oleg Zhurakousky
 */
public class HelloWorldDemo {

	public static void main(String[] args) {
		HelloWorldDemo demo = new HelloWorldDemo();
		AbstractApplicationContext applicationContext = new ClassPathXmlApplicationContext("helloWorldDemo.xml", HelloWorldDemo.class);	
		demo.performDemo(applicationContext);
		applicationContext.stop();
	}


	public void performDemo(ApplicationContext applicationContext) {
		ChannelResolver channelResolver = new BeanFactoryChannelResolver(applicationContext);
		MessageChannel inputChannel = channelResolver.resolveChannelName("inputChannel");
		PollableChannel outputChannel = (PollableChannel) channelResolver.resolveChannelName("outputChannel");
		inputChannel.send(new StringMessage("World"));
		System.out.println("==> HelloWorldDemo: " + outputChannel.receive(0).getPayload());
	}

}
